#include <wx/wx.h>
#include <wx/filename.h>
#include <wx/textdlg.h>
#include <wx/colordlg.h>
#include <wx/config.h>
#include <wx/gbsizer.h>
#include <wx/fileconf.h>
#include <wx/wfstream.h>

#include "App.h"
#include "Frame.h"
#include "MyTimeCtrl.h"

#include <atlbase.h>
#include <atlconv.h>
#include <comutil.h>
#include <ole2.h>
#include <dshow.h>

using namespace std;
using namespace DataAccess;

IMPLEMENT_CLASS(Frame, wxFrame)

//Table of the events and the functions that call
BEGIN_EVENT_TABLE(Frame, wxFrame)
	EVT_MENU(Quit, Frame::OnQuit)
	EVT_MENU(About_Menu, Frame::OnAbout)
	EVT_MENU(Background_Color, Frame::OnBackgroundColor)
	EVT_MENU(Font_Color, Frame::OnFontColor)
	EVT_MENU(Physicians_Number, Frame::OnPhysicianNumber)
	EVT_MENU(Change_Name, Frame::OnNameChange)
	EVT_MENU(Export_Excel, Frame::OnExportExcel)
	EVT_BUTTON(Submit_Button_Clicked, Frame::OnSubmitButtonClick)
	EVT_BUTTON(Continue_Button_Clicked, Frame::OnContinueButtonClick)
	EVT_BUTTON(Glucose_Button_Clicked, Frame::OnGlucoseButtonClick)
	EVT_BUTTON(Pedometer_Button_Clicked, Frame::OnPedometerButtonClick)
	EVT_BUTTON(Physician_Button_Clicked, Frame::OnPhysicianButtonClick)
	EVT_BUTTON(Name_Button_Clicked, Frame::OnNameButtonClick)
END_EVENT_TABLE()


//Constructor for the Frame class.
Frame::Frame(const wxString& title, const wxPoint& pos, const wxSize& size, long style)
: wxFrame((wxFrame *)NULL, wxID_ANY, title, pos, size, style), _view(0)
{
	//menu item boolean variables, used for differentiating between script and menu item events
	_physicianMenuItemClicked = false;
	_nameMenuItemClicked = false;
	_endOfProgram = false;
	_allowScriptMenuEvents = false;

	//reads from the config color to determine saved/preset settings
	wxConfigBase *pConfig = wxConfigBase::Get();
    long red, green, blue;
	pConfig->Read(_T("Window/BackColor-R"), &red, 255);
	pConfig->Read(_T("Window/BackColor-G"), &green, 255);
	pConfig->Read(_T("Window/BackColor-B"), &blue, 255);

	wxColour backColour = wxColour(red, green, blue, 1.0);

	pConfig->Read(_T("Window/ForeColor-R"), &red, 0);
	pConfig->Read(_T("Window/ForeColor-G"), &green, 0);
	pConfig->Read(_T("Window/ForeColor-B"), &blue, 0);

	wxColour foreColour = wxColour(red, green, blue, 1.0);

	//creates file menu items
	wxMenu *fileMenu = new wxMenu(wxEmptyString, wxMENU_TEAROFF);
	fileMenu->Append(Export_Excel, _("&Export to Excel\tAlt+E"), _("Shows your data in excel format."));
	fileMenu->AppendSeparator();
	fileMenu->Append(Quit, _("E&xit\tAlt+X"), _("Quit this program"));

	//creates setting menu items
	wxMenu *settingsMenu = new wxMenu(wxEmptyString, wxMENU_TEAROFF);
	settingsMenu->Append(Physicians_Number, _("Physicians &Number\tAlt+N"), _("Change the entered Physicians number."));
	settingsMenu->Append(Change_Name, _("&Change name\tAlt+C"), _("Change the entered name for yourself."));
	settingsMenu->AppendSeparator();
	settingsMenu->Append(Background_Color, _("&Background color\tAlt+B"), _("Change background color."));
	settingsMenu->Append(Font_Color, _("&Font color\tAlt+F"), _("Change font color."));
	
	//creates help menu items
	wxMenu *helpMenu = new wxMenu(wxEmptyString, wxMENU_TEAROFF);
	helpMenu->Append(About_Menu, _("&About...\tCtrl+A"), _("Show about dialog."));

	//creates the menu bar for the frame
	wxMenuBar *menuBar = new wxMenuBar();
	menuBar->Append(fileMenu, _("&File"));
	menuBar->Append(settingsMenu, _("&Settings"));
	menuBar->Append(helpMenu, _("&Help"));

	SetMenuBar(menuBar);

	_panel = new wxPanel(this, wxID_ANY);

	//creates a sizer to organize components
	wxGridBagSizer *gridSizer = new wxGridBagSizer();
	gridSizer->AddGrowableCol(0);
	gridSizer->AddGrowableRow(0);
	gridSizer->AddGrowableCol(1);
	gridSizer->AddGrowableRow(1);
	gridSizer->AddGrowableCol(2);
	gridSizer->AddGrowableRow(2);
	gridSizer->AddGrowableCol(3);
	gridSizer->AddGrowableRow(3);
	gridSizer->AddGrowableCol(4);
	gridSizer->AddGrowableRow(4);

	//create view that deals with the face graphics, if it fails exit program
	try
	{
		_view = new View(_panel, this, wxID_ANY, wxDefaultPosition, wxDefaultSize);
		gridSizer->Add(_view, wxGBPosition(0, 0), wxGBSpan(4, 4),  wxEXPAND | wxRIGHT, 2);
	}
	catch(...)
	{
		wxMessageBox(_T("Unable to create OpenGL window, you might have a problem with your video card drivers"), _T("Exception"));
		Close(true);
	}
	
	//creates the text box that will display a text version of what the Talking Head is saying
	_textBox = new wxTextCtrl(_panel, wxID_ANY, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxTE_WORDWRAP | wxTE_MULTILINE);
	_textBox->SetEditable(false);
	_textBox->SetBackgroundColour(backColour);
	_textBox->SetForegroundColour(foreColour);
	_textBox->SetFont(wxFont(15, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	gridSizer->Add(_textBox, wxGBPosition(4, 0), wxGBSpan(1, 5),  wxEXPAND);
	
	//creates a panel that will hold all of the user response components
	_inputPanel = new wxPanel(_panel, wxID_ANY, wxDefaultPosition, wxDefaultSize);
	_inputPanel->SetBackgroundColour(backColour);
	gridSizer->Add(_inputPanel, wxGBPosition(0, 4), wxGBSpan(4, 1),  wxEXPAND);

	_inputSizer = new wxBoxSizer(wxVERTICAL);

	_inputPanel->SetSizer(_inputSizer);
	_panel->SetSizer(gridSizer);
    _panel->SetAutoLayout(TRUE);
}

Frame::~Frame()
{
}

#pragma region SetInitialFacepaths

void Frame::SetFAP(const wxString& filename)
{
	_view->SetInitialFap((const char*)filename.c_str());
}

void Frame::SetFDP(const wxString& filename, const wxString& path)
{
	_view->SetInitialFdp((const char*)filename.c_str(), (const char*)path.c_str());
}

#pragma endregion

#pragma region ExcelWrappers

//A wrapper that simplifies most of the low-level details involved with using IDispatch directly.
void AutoWrap(int autoType, VARIANT *pvResult, IDispatch *pDisp, LPOLESTR ptName, int cArgs...)
{
    va_list marker;
    va_start(marker, cArgs);

    if(!pDisp)
	{
        MessageBox(NULL, "NULL IDispatch passed to AutoWrap()", "Error", 0x10010);
        _exit(0);
    }

    DISPPARAMS dp = { NULL, NULL, 0, 0 };
    DISPID dispidNamed = DISPID_PROPERTYPUT;
    DISPID dispID;
    HRESULT hr;
    char buf[200];
    char szName[200];

    WideCharToMultiByte(CP_ACP, 0, ptName, -1, szName, 256, NULL, NULL);
    
    hr = pDisp->GetIDsOfNames(IID_NULL, &ptName, 1, LOCALE_USER_DEFAULT, &dispID);
    if(FAILED(hr))
	{
        sprintf(buf, "IDispatch::GetIDsOfNames(\"%s\") failed w/err 0x%08lx", szName, hr);
        MessageBox(NULL, buf, "AutoWrap()", 0x10010);
        _exit(0);
    }
    
    VARIANT *pArgs = new VARIANT[cArgs+1];
    for(int i=0; i<cArgs; i++)
	{
        pArgs[i] = va_arg(marker, VARIANT);
    }
    
    dp.cArgs = cArgs;
    dp.rgvarg = pArgs;
   
    if(autoType & DISPATCH_PROPERTYPUT)
	{
        dp.cNamedArgs = 1;
        dp.rgdispidNamedArgs = &dispidNamed;
    }
    
    hr = pDisp->Invoke(dispID, IID_NULL, LOCALE_SYSTEM_DEFAULT, autoType, &dp, pvResult, NULL, NULL);
    if(FAILED(hr))
	{
        sprintf(buf, "IDispatch::Invoke(\"%s\"=%08lx) failed w/err 0x%08lx", szName, dispID, hr);
        MessageBox(NULL, buf, "AutoWrap()", 0x10010);
        _exit(0);
    }

    va_end(marker);
    
    delete [] pArgs;
}

//A wrapper that simplifies setting the font in an excel spreadsheet
void SetFont(IDispatch *sheet, VARIANT variable, string fontRangeName, string fontName, int fontSize, bool fontBold)
{
	//create the range of cells to change
	IDispatch *fontRange;
	{
		VARIANT parm;
		parm.vt = VT_BSTR;
		parm.bstrVal = _com_util::ConvertStringToBSTR(fontRangeName.c_str());

		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, sheet, L"Range", 1, parm);
		VariantClear(&parm);

		fontRange = result.pdispVal;
	}

	AutoWrap(DISPATCH_PROPERTYPUT, NULL, fontRange, L"Value", 1, variable);

	//create a font type
	IDispatch *fontSelection;
    {
        VARIANT result;
        VariantInit(&result);
        AutoWrap(DISPATCH_PROPERTYGET, &result, fontRange, L"Font", 0);
        fontSelection = result.pdispVal;
    }

	//set the font type you created
    {
		VARIANT font;
		font.vt = VT_BSTR;
		font.bstrVal = _com_util::ConvertStringToBSTR(fontName.c_str());

        AutoWrap(DISPATCH_PROPERTYPUT, NULL, fontSelection, L"Name", 1, font);
        VARIANT dummy;
        dummy.vt = VT_I4;
        dummy.lVal = fontSize;
        AutoWrap(DISPATCH_PROPERTYPUT, NULL, fontSelection, L"Size", 1, dummy);
        dummy.lVal = fontBold?1:0;
        AutoWrap(DISPATCH_PROPERTYPUT, NULL, fontSelection, L"Bold", 1, dummy);
    }

	//set the alignment
	{
        VARIANT x;
        x.vt = VT_I4;
        x.lVal = -4108;

        AutoWrap(DISPATCH_PROPERTYPUT, NULL, fontRange, L"HorizontalAlignment", 1, x);
    }

	fontRange->Release();
	fontSelection->Release();
}

//A wrapper that simplifies merging cells in an excel spreadsheet
void MergeCells(IDispatch *sheet, string headerName, string initialRange, string mergedRange)
{
	//create a variable to store the name of the merged cells
	VARIANT header;
	header.vt = VT_ARRAY | VT_VARIANT;
	{
		SAFEARRAYBOUND sab[2];
		sab[0].lLbound = 1; sab[0].cElements = 1;
		sab[1].lLbound = 1; sab[1].cElements = 1;
		header.parray = SafeArrayCreate(VT_VARIANT, 2, sab);
	}

	//create the name that will be stored
	VARIANT Name;
	Name.vt = VT_BSTR;
	Name.bstrVal = _com_util::ConvertStringToBSTR(headerName.c_str());
	long indices[] = {1, 1};
	SafeArrayPutElement(header.parray, indices, (void *)&Name);

	//create the range of cells to change
	IDispatch *headerRange;
	{
		VARIANT parm;
		parm.vt = VT_BSTR;
		parm.bstrVal = _com_util::ConvertStringToBSTR(initialRange.c_str());

		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, sheet, L"Range", 1, parm);
		VariantClear(&parm);

		headerRange = result.pdispVal;
	}

	//merge the range of cells
	{
        VARIANT x;
        x.vt = VT_I4;
        x.lVal = -4108;

        AutoWrap(DISPATCH_PROPERTYPUT, NULL, headerRange, L"MergeCells", 1, x);
    }

	//set the merged cells to the stored name
	{
		VARIANT parm;
		parm.vt = VT_BSTR;
		parm.bstrVal = _com_util::ConvertStringToBSTR("A1");

		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, sheet, L"Range", 1, parm);
		VariantClear(&parm);

		headerRange = result.pdispVal;
	}

	AutoWrap(DISPATCH_PROPERTYPUT, NULL, headerRange, L"Value", 1, header);

	::SetFont(sheet, header, mergedRange, "Times New Roman", 14, true);

	headerRange->Release();
}

#pragma endregion

#pragma region ClearInputPanelandTextBox

//Clears the user input panel of any buttons/text boxes
void Frame::ClearInputPanel()
{
	//loop through and destroy each child component
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		_inputPanel->GetChildren()[x]->Destroy();
		x--;
	}
}

//Clears the text box of text and then appends the new text.
void Frame::SetTextBoxText(const string& text)
{
	_textBox->Clear();
	_textBox->AppendText(text);
	_textBox->SetInsertionPoint(0);
}

#pragma endregion

#pragma region SetupForScriptEvents

void Frame::MakeHeadTalk(const string& script)
{
	//_view->loadSMILScript(script);
	//_view->startPlayback();
}

//Adds a way for the user to enter and then submit their pedometer numbers for the day.
void Frame::SetupPedometerReading()
{
	//create a text box for pedometer input
	wxTextCtrl* pedoMeterInput = new wxTextCtrl(_inputPanel, wxID_ANY);
	pedoMeterInput->SetEditable(true);
	pedoMeterInput->SetName(_("PEDOMETER_INPUT"));
	pedoMeterInput->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a button
	wxButton* submitButton = new wxButton(_inputPanel, Pedometer_Button_Clicked, "Submit");
	submitButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	submitButton->SetName(_("SUBMIT_BUTTON"));

	_inputSizer->Add(pedoMeterInput, 0, wxALIGN_CENTER);
	_inputSizer->Add(submitButton, 0, wxALIGN_CENTER);
	_inputPanel->Layout();

	pedoMeterInput->SetFocus();
}

//Adds a way for the user to enter and then submit their glucose numbers for the day.
void Frame::SetupGlucoseReading(const MessageDataTable* glucoseMessagesTable)
{
	//glucose scripts are stored differently than other scripts, you need both messages from a GroupControl instead of a random one
	MessageDataRow* previousMessage = new MessageDataRow(glucoseMessagesTable->GetDataRows().at(0));

	//create a text box to display the enter time message
	wxTextCtrl* glucoseTimeMessage = new wxTextCtrl(_inputPanel, wxID_ANY, _scriptEvents->ReplaceTextVariables(previousMessage->GetText()), wxDefaultPosition, wxDefaultSize, wxTE_WORDWRAP | wxTE_MULTILINE);
	glucoseTimeMessage->SetEditable(false);
	glucoseTimeMessage->SetName("GLUCOSE_TIME_MESSAGE");
	glucoseTimeMessage->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	wxString dateTime = MyTimeCtrl::GetCurrentTime();

	//midnight is represented as 00:XX so I change it to 12:XX
	if(dateTime.SubString(0, 1) == "00")
	{
		char hour1 = 49, hour2 = 50;
		dateTime[0] = hour1;
		dateTime[1] = hour2;
	}

	//create the time selection box
	MyTimeCtrl* glucoseTimeSelect = new MyTimeCtrl(_inputPanel, wxID_ANY, dateTime);
	glucoseTimeSelect->SetName("GLUCOSE_TIME_SELECT");
	glucoseTimeSelect->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a text box to display the enter glucose number message
	wxTextCtrl* glucoseNumberMessage = new wxTextCtrl(_inputPanel, wxID_ANY, _scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()), wxDefaultPosition, wxDefaultSize, wxTE_WORDWRAP | wxTE_MULTILINE);
	glucoseNumberMessage->SetEditable(false);
	glucoseNumberMessage->SetName("GLUCOSE_NUMBER_MESSAGE");
	glucoseNumberMessage->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a text box for glucose number input
	wxTextCtrl* glucoseNumberInput = new wxTextCtrl(_inputPanel, wxID_ANY);
	glucoseNumberInput->SetEditable(true);
	glucoseNumberInput->SetName("GLUCOSE_NUMBER_INPUT");
	glucoseNumberInput->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a button
	wxButton* submitButton = new wxButton(_inputPanel, Glucose_Button_Clicked, "Submit");
	submitButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	stringstream convert;
	convert << previousMessage->GetID();
	submitButton->SetName(convert.str());

	_inputSizer->Add(glucoseTimeMessage, 0, wxALIGN_CENTER | wxEXPAND);
	_inputSizer->Add(glucoseTimeSelect, 0, wxALIGN_CENTER | wxEXPAND);
	_inputSizer->Add(glucoseNumberMessage, 0, wxALIGN_CENTER | wxEXPAND);
	_inputSizer->Add(glucoseNumberInput, 0, wxALIGN_CENTER | wxEXPAND);
	_inputSizer->Add(submitButton, 0, wxALIGN_CENTER);
	_inputPanel->Layout();

	glucoseNumberInput->SetFocus();
}

//Adds a way for the user to enter and then submit their physicians number.
void Frame::SetupPhysiciansNumber()
{
	//create a text box for physician number input
	wxTextCtrl* physiciansInput = new wxTextCtrl(_inputPanel, wxID_ANY);
	physiciansInput->SetEditable(true);
	physiciansInput->SetName(_("PHYSICIANS_INPUT"));
	physiciansInput->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a button
	wxButton* submitButton = new wxButton(_inputPanel, Physician_Button_Clicked, "Submit");
	submitButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	submitButton->SetName(_("SUBMIT_BUTTON"));

	_inputSizer->Add(physiciansInput, 0, wxALIGN_CENTER);
	_inputSizer->Add(submitButton, 0, wxALIGN_CENTER);
	_inputPanel->Layout();

	physiciansInput->SetFocus();
}

//Adds a way for the user to repsond to a question asked by the Talking Head.
void Frame::AddUserResponse(const MessageDataTable* messageDataTable)
{
	stringstream convert;
	for(int x=0; x< messageDataTable->GetDataRows().size(); x++)
	{
		//create a radio button for user responses
		wxRadioButton* radioButton = new wxRadioButton(_inputPanel, wxID_ANY, messageDataTable->GetDataRows().at(x).GetText());
		radioButton->SetValue(FALSE);
		radioButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
		convert << messageDataTable->GetDataRows().at(x).GetID();
		radioButton->SetName(convert.str());
		convert.str("");

		_inputSizer->Add(radioButton, 0, wxALIGN_LEFT);
	}

	//create a button
	wxButton* submitButton = new wxButton(_inputPanel, Submit_Button_Clicked, "Submit");
	submitButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	submitButton->SetName(_("SUBMIT_BUTTON"));

	_inputSizer->Add(submitButton, 0, wxALIGN_CENTER);

	_inputPanel->Layout();
}

//Adds a way for the user to continue after there is a pause in the script.
void Frame::AddContinueButton()
{
	//create a button
	wxButton* continueButton = new wxButton(_inputPanel, Continue_Button_Clicked, "Continue");
	continueButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	stringstream convert;
	convert << _scriptEvents->GetCurrentMessage()->GetID();
	continueButton->SetName(convert.str());

	_inputSizer->Add(continueButton, 0, wxALIGN_CENTER);
	_inputPanel->Layout();
}

//Adds a way for the user to enter and then submit their name so the Talking Head knows what to address them by.
void Frame::SetupNamePrompt()
{
	//create a text box for user name input
	wxTextCtrl* nameInput = new wxTextCtrl(_inputPanel, wxID_ANY);
	nameInput->SetEditable(true);
	nameInput->SetName(_("NAME_INPUT"));
	nameInput->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));

	//create a button
	wxButton* submitButton = new wxButton(_inputPanel, Name_Button_Clicked, "Submit");
	submitButton->SetFont(wxFont(12, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
	submitButton->SetName(_("SUBMIT_BUTTON"));

	_inputSizer->Add(nameInput, 0, wxALIGN_CENTER);
	_inputSizer->Add(submitButton, 0, wxALIGN_CENTER);
	_inputPanel->Layout();

	nameInput->SetFocus();
}

#pragma endregion

#pragma region ButtonClickEvents

//Called when the submit button event is triggered by a button press.
void Frame::OnSubmitButtonClick(wxCommandEvent &event)
{
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		//find the radio buttons (they are not named SUBMIT_BUTTON
		if(_inputPanel->GetChildren()[x]->GetName() != "SUBMIT_BUTTON")
		{
			wxRadioButton* radioButton = ((wxRadioButton*)_inputPanel->GetChildren()[x]);
			//find the radio button that has been clicked
			if(radioButton->GetValue() == true)
			{
				//set-up next script based on the radio button selected
				(new ResponseHistoryTableAdapter())->InsertResponseHistory(_scriptEvents->GetDataBase(), radioButton->GetLabel().c_str());

				int messageID = atoi(radioButton->GetName());
				_scriptEvents->SetCurrentMessage((new MessageTableAdapter())->FillByID(_scriptEvents->GetDataBase(), messageID));

				_scriptEvents->LoadNextScene();
				break;
			}
		}
	}
}

//Called when the continue button event is triggered by a button press.
void Frame::OnContinueButtonClick(wxCommandEvent &event)
{
	//set-up next script
	wxButton* button = (wxButton*)_inputPanel->GetChildren()[0];
	
	int messageID = atoi(button->GetName());
	_scriptEvents->SetCurrentMessage((new MessageTableAdapter())->FillByID(_scriptEvents->GetDataBase(), messageID));

	_scriptEvents->LoadNextScene();
}

//Called when the glucose button event is triggered by a button press.
void Frame::OnGlucoseButtonClick(wxCommandEvent &event)
{
	MyTimeCtrl* glucoseTimeSelect;
	wxTextCtrl* glucoseNumberInput;
	int previousMessageID;
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		//if the child is the component to select the time of day
		if(_inputPanel->GetChildren()[x]->GetName() == "GLUCOSE_TIME_SELECT")
		{
			glucoseTimeSelect = ((MyTimeCtrl*)_inputPanel->GetChildren()[x]);
		}
		//if the child is the compoent to input your glucose number
		else if(_inputPanel->GetChildren()[x]->GetName() == "GLUCOSE_NUMBER_INPUT")
		{
			glucoseNumberInput = ((wxTextCtrl*)_inputPanel->GetChildren()[x]);
		}
		//if the child is neither of the above
		else if(_inputPanel->GetChildren()[x]->GetName() != "GLUCOSE_NUMBER_MESSAGE" && _inputPanel->GetChildren()[x]->GetName() != "GLUCOSE_TIME_MESSAGE")
		{
			previousMessageID = atoi(_inputPanel->GetChildren()[x]->GetName());
		}
	}
	
	int glucoseNumber = atoi(glucoseNumberInput->GetValue());
	//if they entered a number and not a string
	if(glucoseNumber > 0)
	{
		int hour = atoi(glucoseTimeSelect->GetValue().SubString(0, 1).c_str());

		wxString time = "";
		//if the time entered is PM, convert it to 12 hour time
		if(glucoseTimeSelect->GetValue().SubString(9, 10) == "PM")
		{
			hour += 12;
			wxString minute = glucoseTimeSelect->GetValue().SubString(3, 4);
			wxString second = glucoseTimeSelect->GetValue().SubString(6, 7);

			stringstream convert;
			convert << hour;

			time.append(convert.str());
			time.append(":");
			time.append(minute);
		}
		//time is already in 12 hour time
		else
		{
			time = glucoseTimeSelect->GetValue().SubString(0, 4);
		}

		(new GlucoseTableAdapter())->InsertGlucose(_scriptEvents->GetDataBase(), glucoseNumber, time.c_str());

		//if the glucose number and time entered are the fourth ones
		if((new MessageLinkTableAdapter())->FillByParent(_scriptEvents->GetDataBase(), _scriptEvents->GetCurrentMessage()->GetID())->GetChild() != (new MessageLinkTableAdapter())->FillByParent(_scriptEvents->GetDataBase(), previousMessageID)->GetChild())
		{
			GlucoseDataTable* table = (new GlucoseTableAdapter())->FillByAll(_scriptEvents->GetDataBase());
			wxString times = "";
			
			int size = (table->GetDataRows().size());
			//loops through the last 4 glucose numbers entered
			for(int x=size-5; x<size-1; x++)
			{
				//if a glucose number is less than 100
				if(table->GetDataRows()[x].GetGlucoseLevel() < 100)
				{
					times.append(table->GetDataRows()[x].GetTime());
					times.append(" ");
				}
			}

			//if a glucose number was less than 100
			if(times != "")
			{
				cerr << times << endl;
				_scriptEvents->SetCurrentMessage((new MessageTableAdapter())->FillByID(_scriptEvents->GetDataBase(), previousMessageID));
			}

			//set-up next script based on if any glucose numbers for the day are less than 100
			this->ClearInputPanel();
			MessageLinkDataRow* nextQuestionLink = (new MessageLinkTableAdapter())->FillByParent(_scriptEvents->GetDataBase(), _scriptEvents->GetCurrentMessage()->GetID());

			GroupControlDataRow* nextGroup = (new GroupControlTableAdapter())->FillByID(_scriptEvents->GetDataBase(), nextQuestionLink->GetChild());
			MessageDataTable* nextMessages = (new MessageTableAdapter())->FillByGroupID(_scriptEvents->GetDataBase(), nextGroup->GetID());

			(new CurrentStateTableAdapter())->InsertCurrentState(_scriptEvents->GetDataBase(), 1, 0, _scriptEvents->GetCurrentMessage()->GetID());

			int random = (rand()%((int)nextMessages->GetDataRows().size()));
			MessageDataRow* currentMessage = new MessageDataRow(nextMessages->GetDataRows()[random]);
			_scriptEvents->SetCurrentMessage(currentMessage);

			//if a glucose number was less than 100
			if(times != "")
			{
				string text = _scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText());
				//replaces (TIMES) with the times for the glucose that were under 100
				int position = text.find("(TIMES)", 0);
				if(position != -1)
				{
					text = text.replace(position, 7, times);
				}

				this->SetTextBoxText(text);

				text = _scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetScript());
				//replaces (TIMES) with the times for the glucose that were under 100
				position = text.find("(TIMES)", 0);
				if(position != -1)
				{
					text = text.replace(position, 7, times);
				}

				this->MakeHeadTalk(text);
			}
			else
			{
				this->SetTextBoxText(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
				this->MakeHeadTalk(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetScript()));
			}
		}
		_scriptEvents->LoadNextScene();
	}
	//an invalid number was entered for glucose
	else
	{
		glucoseNumberInput->Clear();
	}
}

//Called when the pedometer button event is triggered by a button press.
void Frame::OnPedometerButtonClick(wxCommandEvent &event)
{
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		//if the child is the compoent to input your pedometer number
		if(_inputPanel->GetChildren()[x]->GetName() == "PEDOMETER_INPUT")
		{
			int pedometerNumber = atoi(((wxTextCtrl*)_inputPanel->GetChildren()[x])->GetValue());

			//if the user inputed a number
			if(pedometerNumber != 0)
			{
				(new ExerciseTableAdapter())->InsertExercise(_scriptEvents->GetDataBase(), pedometerNumber);

				GroupControlDataRow* nextGroup;
				//if the number entered is less than 10000
				if(pedometerNumber < 10000)
				{
					nextGroup = (new GroupControlTableAdapter())->FillByEntry(_scriptEvents->GetDataBase(), 9);

				}
				//if the number entered is greater than or equal to 10000
				else
				{
					nextGroup = (new GroupControlTableAdapter())->FillByEntry(_scriptEvents->GetDataBase(), 8);
				}

				//set-up next script based on if the pedometer number is less than 10000
				MessageDataTable* pedometerResponsesTable = (new MessageTableAdapter())->FillByGroupID(_scriptEvents->GetDataBase(), nextGroup->GetID());

				int random = (rand()%((int)pedometerResponsesTable->GetDataRows().size()));

				_scriptEvents->SetCurrentMessage(new MessageDataRow(pedometerResponsesTable->GetDataRows().at(random)));

				_scriptEvents->LoadContinuedMessage();
			}
			//if the user didn't input a number
			else
			{
				((wxTextCtrl*)_inputPanel->GetChildren()[x])->Clear();
			}
			break;
		}
	}
}

//Called when the physician button event is triggered by a button press.
void Frame::OnPhysicianButtonClick(wxCommandEvent &event)
{
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		//if the child is the compoent to input your physican's number
		if(_inputPanel->GetChildren()[x]->GetName() == "PHYSICIANS_INPUT")
		{
			string physicianNumber = ((wxTextCtrl*)_inputPanel->GetChildren()[x])->GetValue();
			//if the user entered something for a number
			if(physicianNumber != "")
			{
				(new ConfigurationTableAdapter())->UpdateConfigurationByKey(_scriptEvents->GetDataBase(), "PHYSICIAN_NUMBER", physicianNumber);
				_scriptEvents->SetPhysicianNumber(physicianNumber);

				//if this function was called because of a menu event
				if(_physicianMenuItemClicked == true)
				{
					_physicianMenuItemClicked = false;

					//set-up next script
					_scriptEvents->SetCurrentMessage((new MessageTableAdapter())->FillByID(_scriptEvents->GetDataBase(), ((new CurrentStateTableAdapter())->FillByAll(_scriptEvents->GetDataBase())->GetCurrentMessageID())));
					
					this->SetTextBoxText(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
					this->MakeHeadTalk(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetScript()));

					_scriptEvents->LoadContinuedMessage();

					return;
				}

				_scriptEvents->LoadNextScene();
			}
			break;
		}
	}
}

//Called when the name button event is triggered by a button press.
void Frame::OnNameButtonClick(wxCommandEvent &event)
{
	for(int x=0; x<_inputPanel->GetChildren().GetCount(); x++)
	{
		//if the child is the compoent to input your name
		if(_inputPanel->GetChildren()[x]->GetName() == "NAME_INPUT")
		{
			string name = ((wxTextCtrl*)_inputPanel->GetChildren()[x])->GetValue();

			//if the user entered something for a name
			if(name != "")
			{
				(new ConfigurationTableAdapter())->UpdateConfigurationByKey(_scriptEvents->GetDataBase(), "PATIENT_NAME", name);
				_scriptEvents->SetPatientName(name);
				
				//if this function was called because of a menu event
				if(_nameMenuItemClicked == true)
				{
					_nameMenuItemClicked = false;

					//set-up next script
					_scriptEvents->SetCurrentMessage((new MessageTableAdapter())->FillByID(_scriptEvents->GetDataBase(), ((new CurrentStateTableAdapter())->FillByAll(_scriptEvents->GetDataBase())->GetCurrentMessageID())));
					
					this->SetTextBoxText(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
					this->MakeHeadTalk(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetScript()));

					_scriptEvents->LoadContinuedMessage();

					return;
				}

				_scriptEvents->LoadNextScene();
			}
			break;
		}
	}
}
#pragma endregion

#pragma region MenuItemEvents

//Called when the frame is to be closed
void Frame::OnQuit(wxCommandEvent& event)
{
	Close(TRUE);
}

//Called when the About menu item is clicked
void Frame::OnAbout(wxCommandEvent& event)
{
	//print-out about information
	wxString msg;
	msg.Printf(_("About Talking Head..."));
	wxMessageBox(_("Talking Head by THT."), msg, wxOK | wxICON_INFORMATION, this);
}

//Called when the background color menu item is clicked
void Frame::OnBackgroundColor(wxCommandEvent& event)
{
	wxColourDialog dlg(this);
	//when the user presses okay
	if(dlg.ShowModal() == wxID_OK)
	{
		//set the color of the components
		_view->SetBackgroundColour(dlg.GetColourData().GetColour());
		_inputPanel->SetBackgroundColour(dlg.GetColourData().GetColour());
		_textBox->SetBackgroundColour(dlg.GetColourData().GetColour());

		//retreive selected color
		long red, green, blue;
		red = dlg.GetColourData().GetColour().Red();
		green = dlg.GetColourData().GetColour().Green();
		blue = dlg.GetColourData().GetColour().Blue();

		//store selected color to config file
		wxFileInputStream st(wxT("talkinghead.ini"));
		wxFileConfig *pConfig = new wxFileConfig(st);
		pConfig->Write(_T("Window/BackColor-R"), red);
		pConfig->Write(_T("Window/BackColor-G"), green);
		pConfig->Write(_T("Window/BackColor-B"), blue);
		pConfig->Save(wxFileOutputStream("talkinghead.ini"));

		this->Refresh();
	}
}

//Called when the font color menu item is clicked
void Frame::OnFontColor(wxCommandEvent& event)
{
	wxColourDialog dlg(this);
	//when the user presses okay
	if(dlg.ShowModal() == wxID_OK)
	{
		//set the color of the font
		_textBox->SetForegroundColour(dlg.GetColourData().GetColour());
		
		//retreive selected font color
		long red, green, blue;
		red = dlg.GetColourData().GetColour().Red();
		green = dlg.GetColourData().GetColour().Green();
		blue = dlg.GetColourData().GetColour().Blue();

		//store selected font color to config file
		wxFileInputStream st(wxT("talkinghead.ini"));
		wxFileConfig *pConfig = new wxFileConfig(st);
		pConfig->Write(_T("Window/ForeColor-R"), red);
		pConfig->Write(_T("Window/ForeColor-G"), green);
		pConfig->Write(_T("Window/ForeColor-B"), blue);
		pConfig->Save(wxFileOutputStream("talkinghead.ini"));

		this->Refresh();
	}
}

//Called when the physician number menu item is clicked
void Frame::OnPhysicianNumber(wxCommandEvent& event)
{
	if(_allowScriptMenuEvents == true)
	{
		_physicianMenuItemClicked = true;

		(new CurrentStateTableAdapter())->InsertCurrentState(_scriptEvents->GetDataBase(), 1, 0, _scriptEvents->GetCurrentMessage()->GetID());

		this->ClearInputPanel();

		//sets the current message to be the script for physician number input
		GroupControlDataRow* physiciansRow = (new GroupControlTableAdapter())->FillByEntry(_scriptEvents->GetDataBase(), 11);
		MessageDataTable* physiciansTable = (new MessageTableAdapter())->FillByGroupID(_scriptEvents->GetDataBase(), physiciansRow->GetID());
	                
		int random = (rand()%((int)physiciansTable->GetDataRows().size()));

		_scriptEvents->SetCurrentMessage(new MessageDataRow(physiciansTable->GetDataRows().at(random)));

		this->SetTextBoxText(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
		this->MakeHeadTalk(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));

		this->SetupPhysiciansNumber();
	}
}

//Called when the name change menu item is clicked
void Frame::OnNameChange(wxCommandEvent& event)
{
	if(_allowScriptMenuEvents == true)
	{
		_nameMenuItemClicked = true;

		(new CurrentStateTableAdapter())->InsertCurrentState(_scriptEvents->GetDataBase(), 1, 0, _scriptEvents->GetCurrentMessage()->GetID());

		this->ClearInputPanel();

		//sets the current message to be the script for name change input
		GroupControlDataRow* nameRow = (new GroupControlTableAdapter())->FillByEntry(_scriptEvents->GetDataBase(), 6);
		MessageDataTable* nameTable = (new MessageTableAdapter())->FillByGroupID(_scriptEvents->GetDataBase(), nameRow->GetID());

		int random = (rand()%((int)nameTable->GetDataRows().size()));

		_scriptEvents->SetCurrentMessage(new MessageDataRow(nameTable->GetDataRows().at(random)));

		this->SetTextBoxText(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
		this->MakeHeadTalk(_scriptEvents->ReplaceTextVariables(_scriptEvents->GetCurrentMessage()->GetText()));
		this->SetupNamePrompt();
	}
}

//Called when the export to excel menu item is clicked
void Frame::OnExportExcel(wxCommandEvent& event)
{
	USES_CONVERSION;
	CoInitialize(NULL);

	//checks to see if excel is installed on the computer
	CLSID clsid;
	HRESULT hr = CLSIDFromProgID(L"Excel.Application", &clsid);
	if(FAILED(hr))
	{
		wxMessageBox(_("You do not have Microsoft Excel installed. Please install it before trying again."), "Exporting Error", wxOK | wxICON_ERROR, this);
		return;
	}

	IDispatch *pXlApp;
	hr = CoCreateInstance(clsid, NULL, CLSCTX_LOCAL_SERVER, IID_IDispatch, (void **)&pXlApp);
	if(FAILED(hr))
	{
		wxMessageBox(_("You do not have Microsoft Excel installed. Please install it before trying again."), "Exporting Error", wxOK | wxICON_ERROR, this);
		return;
	}

	//creates the workbooks variable
	IDispatch *pXlBooks;
	{
		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, pXlApp, L"Workbooks", 0);
		pXlBooks = result.pdispVal;
	}

	//adds a new workbook
	IDispatch *pXlBook;
	{
		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, pXlBooks, L"Add", 0);
		pXlBook = result.pdispVal;
	}

	//gets the active sheet from the workbook
	IDispatch *pXlSheet;
	{
		VARIANT result;
		VariantInit(&result);
		AutoWrap(DISPATCH_PROPERTYGET, &result, pXlApp, L"ActiveSheet", 0);
		pXlSheet = result.pdispVal;
	}

	//sets-up the date for the glucose numbers
	int glucoseColumn = 3, exerciseColumn = 2;
	GlucoseDataTable* glucoseTable = (new GlucoseTableAdapter())->FillByAll(_scriptEvents->GetDataBase());
	VARIANT glucose;
	glucose.vt = VT_ARRAY | VT_VARIANT;
	{
		SAFEARRAYBOUND sab[2];
		sab[0].lLbound = 1; sab[0].cElements = glucoseTable->GetDataRows().size();
		sab[1].lLbound = 1; sab[1].cElements = glucoseColumn;
		glucose.parray = SafeArrayCreate(VT_VARIANT, 2, sab);
	}

	//stores the glucose data into a Variant
	int count=1;
	for(int row=glucoseTable->GetDataRows().size(); row>=1; row--)
	{
		GlucoseDataRow* glucoseRow = new GlucoseDataRow(glucoseTable->GetDataRows()[row-1]);
		VARIANT Date, Time, Level;
		Date.vt = VT_BSTR;
		Date.bstrVal = _com_util::ConvertStringToBSTR(glucoseRow->GetDate().c_str());
		long indices1[] = {count, 1};
		SafeArrayPutElement(glucose.parray, indices1, (void *)&Date);

		Time.vt = VT_BSTR;
		Time.bstrVal = _com_util::ConvertStringToBSTR(glucoseRow->GetTime().c_str());
		long indices2[] = {count, 2};
		SafeArrayPutElement(glucose.parray, indices2, (void *)&Time);

		Level.vt = VT_INT;
		Level.intVal = glucoseRow->GetGlucoseLevel();
		long indices3[] = {count, 3};
		SafeArrayPutElement(glucose.parray, indices3, (void *)&Level);
		count++;
	}

	//sets the range for the glucose data
	string range = "A3:C";
	stringstream convert;
	convert << glucoseTable->GetDataRows().size() + 2;
	range.append(convert.str());

	::SetFont(pXlSheet, glucose, range, "Times New Roman", 12, false);

	//sets-up the date for the exercise numbers
	ExerciseDataTable* exerciseTable = (new ExerciseTableAdapter())->FillByAll(_scriptEvents->GetDataBase());
	VARIANT exercise;
	exercise.vt = VT_ARRAY | VT_VARIANT;
	{
		SAFEARRAYBOUND sab[2];
		sab[0].lLbound = 1; sab[0].cElements = (exerciseTable->GetDataRows().size() * 4);
		sab[1].lLbound = 1; sab[1].cElements = exerciseColumn;
		exercise.parray = SafeArrayCreate(VT_VARIANT, 2, sab);
	}

	//stores the exercise data into a Variant
	count=1;
	for(int row=(exerciseTable->GetDataRows().size() * 4); row>=1; row-=4)
	{
		ExerciseDataRow* exerciseRow = new ExerciseDataRow(exerciseTable->GetDataRows()[(row/4)-1]);
		VARIANT Date, Steps;
		Date.vt = VT_BSTR;
		Date.bstrVal = _com_util::ConvertStringToBSTR(exerciseRow->GetDate().c_str());
		long indices1[] = {count, 1};
		SafeArrayPutElement(exercise.parray, indices1, (void *)&Date);

		Steps.vt = VT_INT;
		Steps.intVal = exerciseRow->GetTotalSteps();
		long indices2[] = {count, 2};
		SafeArrayPutElement(exercise.parray, indices2, (void *)&Steps);
		count+=4;
	}

	//sets the range for the exercise data
	range = "E3:F";
	convert.str("");
	convert << (exerciseTable->GetDataRows().size() * 4) + 2;
	range.append(convert.str());
	
	::SetFont(pXlSheet, exercise, range, "Times New Roman", 12, false);

	//sets-up the header for the glucose data
	VARIANT glucoseNameVar;
	glucoseNameVar.vt = VT_ARRAY | VT_VARIANT;
	{
		SAFEARRAYBOUND sab[2];
		sab[0].lLbound = 1; sab[0].cElements = 1;
		sab[1].lLbound = 1; sab[1].cElements = 3;
		glucoseNameVar.parray = SafeArrayCreate(VT_VARIANT, 2, sab);
	}

	//stores the glucose header data into Variants
	VARIANT Date, Time, Level;
	Date.vt = VT_BSTR;
	Date.bstrVal = _com_util::ConvertStringToBSTR("Date");
	long indices1[] = {1, 1};
	SafeArrayPutElement(glucoseNameVar.parray, indices1, (void *)&Date);

	Time.vt = VT_BSTR;
	Time.bstrVal = _com_util::ConvertStringToBSTR("Time");
	long indices2[] = {1, 2};
	SafeArrayPutElement(glucoseNameVar.parray, indices2, (void *)&Time);

	Level.vt = VT_BSTR;
	Level.bstrVal = _com_util::ConvertStringToBSTR("Level");
	long indices3[] = {1, 3};
	SafeArrayPutElement(glucoseNameVar.parray, indices3, (void *)&Level);

	::SetFont(pXlSheet, glucoseNameVar, "A2:C2", "Times New Roman", 12, true);

	//sets-up the header for the exercise data
	VARIANT exerciseNameVar;
	exerciseNameVar.vt = VT_ARRAY | VT_VARIANT;
	{
		SAFEARRAYBOUND sab[2];
		sab[0].lLbound = 1; sab[0].cElements = 1;
		sab[1].lLbound = 1; sab[1].cElements = 2;
		exerciseNameVar.parray = SafeArrayCreate(VT_VARIANT, 2, sab);
	}

	//stores the exercise header data into Variants
	VARIANT Steps;
	Date.vt = VT_BSTR;
	Date.bstrVal = _com_util::ConvertStringToBSTR("Date");
	long indices4[] = {1, 1};
	SafeArrayPutElement(exerciseNameVar.parray, indices4, (void *)&Date);

	Steps.vt = VT_BSTR;
	Steps.bstrVal = _com_util::ConvertStringToBSTR("Steps");
	long indices5[] = {1, 2};
	SafeArrayPutElement(exerciseNameVar.parray, indices5, (void *)&Steps);

	::SetFont(pXlSheet, exerciseNameVar, "E2:F2", "Times New Roman", 12, true);

	//merges the header cells for the data
	::MergeCells(pXlSheet, "Glucose", "A1:C1", "A1");
	::MergeCells(pXlSheet, "Exercise", "E1:F1", "E1");

	{
		VARIANT x;
		x.vt = VT_I4;
		x.lVal = 1;
		AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"Visible", 1, x);
	}

	pXlSheet->Release();
	pXlBook->Release();
	pXlBooks->Release();
	pXlApp->Release();
	VariantClear(&glucose);
	VariantClear(&exercise);

	CoUninitialize();
}

#pragma endregion

#pragma region Video Events

//Called when a video needs to be played
void Frame::PlayVideo(string videoName)
{
	_view->Hide();
	_ctrl = new wxMediaCtrl(_panel, wxID_ANY, videoName, wxPoint(0, 0), _panel->GetSize(), 0, wxMEDIABACKEND_WMP10);

	WaitTimer* timer = new WaitTimer(this, _ctrl);
	timer->Start(300, true);
}

//Called when the intro video is done playing
void Frame::AfterIntroVideo()
{
	delete _ctrl;
	_view->Show();
	_scriptEvents->CheckState();
	_allowScriptMenuEvents = true;
}

//Called when the end video is done playing
void Frame::AfterEndingVideo()
{
	delete _ctrl;
	this->ClearInputPanel();
	this->SetTextBoxText("");
	_allowScriptMenuEvents = true;
}

#pragma endregion